/*
 * ROHM_Stepper.h - ROHM Stepper Motor Drivers library for Arduino - Version 2.0.0
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

// initial release 1.0.0
// updated release 1.1.0
// updated release 2.0.0 including BD63510AEFV, BD63520AEFV, BD63524AEFV

#ifndef ROHM_Steppers_h
#define ROHM_Steppers_h

// Motor Driver IC's types 

#if defined  BD63710AEFV
#define TYPE1
#endif

#if defined  BD63715AEFV
#define TYPE1
#endif

#if defined  BD63720AEFV
#define TYPE1
#endif

#if defined  BD6425EFV
#define TYPE1
#endif

#if defined  BD63843EFV
#define TYPE2
#endif

#if defined  BD63847EFV
#define TYPE2
#endif

#if defined  BD63510AEFV
#define TYPE3
#endif

#if defined  BD63520AEFV
#define TYPE3
#endif

#if defined  BD63524AEFV
#define TYPE4
#endif


// TYPE1 BD63710AEFV, BD63715AEFV, BD63720AEFV, BD6425EFV
// TYPE2 BD63843EFV, BD63847EFV
// TYPE3 BD63510AEFV, BD63520AEFV
// TYPE3 BD63524AEFV


// Arduino pin related to IC driver pin

// ONE or TWO boards configuration available

#define  pinENABLE1   2    //SLAVE BOARD pins assigning
#define  pinPS1       A0
#define  pinMODE11    4
#define  pinMODE01    5
#define  pinCW_CCW1   12
#define  pinCLK1      9
#define  pinTEST11    8

#define  pinENABLE0   3    // MASTER BOARD pins assigning
#define  pinPS0       A1
#define  pinMODE10    7
#define  pinMODE00    6
#define  pinCW_CCW0   13
#define  pinCLK0      10
#define  pinTEST10    7

// Driver IC states and mnemonics TYPE1
#if defined TYPE1
#define FULL_STEP  		1  //  
#define HALF_STEP_A   	5  //  
#define HALF_STEP_B   	6  //
#define QUARTER_STEP   	4  //
#endif

// Driver IC states and mnemonics TYPE2
#if defined TYPE2
#define FULL_STEP  		1  //  
#define HALF_STEP   	2  // 
#define EIGHTH_STEP		8  //
#define SIXTEENTH_STEP  16 //
#endif

// Driver IC states and mnemonics TYPE3
#if defined TYPE3
#define FULL_STEP  		1  //  
#define HALF_STEP   	2  // 
#define QUARTER_STEP   	4  //
#define SIXTEENTH_STEP  16 //
#endif

// Driver IC states and mnemonics TYPE4
#if defined TYPE4
#define FULL_STEP  		1  //  
#define HALF_STEP   	2  // 
#define QUARTER_STEP   	4  //
#define EIGHTH_STEP		8  //
#endif

// Driver IC states and mnemonics TYPE1, TYPE2, TYPE3, TYPE4
#define OPEN			0  //
#define RESET			0  //
#define ACTIVE			1  //
#define CW              0  //
#define CCW             1  //

// half period for clocks by default
#define HPdef 1000     //microseconds             

// numbers for boards 
#define ONE   1 //
#define TWO   2 //
#define NOBRD 0 //

// declaration of the class is here
class ROHM_Stepper 
{
public:

	ROHM_Stepper( int BoardNumber);  // ONE, TWO
	
// Stepping motors low level pins API

    // master board ( single board configuration )
    void ENABLE(int en);        // pin 20
    void PS    (int ps);        // pin 14
    void MODE  (int mode);      // pin 18 MODE0 , pin 19 MODE1
    void CW_CCW(int dir);       // pin 16
    void CLK   (int clk);       // pin 15 
	
    // master board in two board configuration
    void ENABLE_M(int  en)  {ENABLE(  en);  };      // pin 20
    void PS_M    (int  ps)  {PS    (  ps);  };      // pin 14
    void MODE_M  (int  mode){MODE  (mode);  };      // pin 18 MODE0 , pin 19 MODE1
    void CW_CCW_M(int  dir) {CW_CCW( dir);  };      // pin 16
    void CLK_M   (int  clk) {CLK   ( clk);  };      // pin 15 (0 - STOP, -1 NOSTOP)		
	
	// slave board ( two boards configuration )
    void ENABLE_S(int en);      // pin 20
    void PS_S    (int ps);      // pin 14
    void MODE_S  (int mode);    // pin 18 MODE0 , pin 19 MODE1
    void CW_CCW_S(int dir);     // pin 16
    void CLK_S   (int clk);     // pin 15 
	
// Clock  emulation	
    void         setCLK_HP(unsigned int hp); // set clock half period for CLK in microsecond 
    unsigned int getCLK_HP();                // return clock half period of CLK in microsecond 	
	void         setCLK_Hz(unsigned int hz); // set clock in Hertz
	unsigned int getCLK_Hz();                // return clock in Hertz
	
// Commnon
	int version(void); // return version of the lib
	
private:

	volatile int steps;   // clocks counter for MASTER board
	volatile int steps_S; // clocks counter for SLAVE board
  
	int  boards;          // available configurations NOBRD ONE TWO
	unsigned int  tHP;    // time value im microseconds for clock half period ( predefined or by CLKHP(hp) ) 

};

ROHM_Stepper::ROHM_Stepper(int brd) // ONE or TWO boards
{	
    if (brd == ONE || brd == TWO)
	{   boards=brd;

		//init for MASTER board
		pinMode(pinENABLE0,     OUTPUT);
		pinMode(pinPS0,     	OUTPUT);
		pinMode(pinMODE10,      OUTPUT);
		pinMode(pinMODE00,      OUTPUT);
		pinMode(pinCW_CCW0,     OUTPUT);
		pinMode(pinCLK0,     	OUTPUT);
		//pinMode(pinTEST10,     OUTPUT);  // we do not use this pin ( CMOS input on board )
		
        // set up by default
		// NOT ACTIVE, RESET, FULL_STEP, CW
	    digitalWrite(pinENABLE0,  OPEN ); // NOT ACTIVE
		digitalWrite(pinPS0,     RESET ); // RESET
		digitalWrite(pinMODE10,   LOW  ); // FULL_STEP
		digitalWrite(pinMODE00,   LOW  );
		digitalWrite(pinCW_CCW0,   CW  ); // CW
		digitalWrite(pinCLK0,     LOW  ); // LOW 
		//digitalWrite(pinTEST10,     );  // we do not use this pin ( CMOS input on board )
		
		tHP = HPdef; // half period by default
		
		if (brd == TWO)
		{
			// init for SLAVE board
			pinMode(pinENABLE1,     OUTPUT);
			pinMode(pinPS1,     	OUTPUT);
			pinMode(pinMODE11,      OUTPUT);
			pinMode(pinMODE01,      OUTPUT);
			pinMode(pinCW_CCW1,     OUTPUT);
			pinMode(pinCLK1,     	OUTPUT);
			// pinMode(pinTEST11,     OUTPUT); // we do not use this pin ( CMOS input on board )
			
			// set up by default
			// NOT ACTIVE, RESET, FULL_STEP, CW
			digitalWrite(pinENABLE1,  OPEN );  // NOT ACTIVE
			digitalWrite(pinPS1,     RESET );  // RESET
			digitalWrite(pinMODE11,   LOW  );  // FULL_STEP
			digitalWrite(pinMODE01,   LOW  );
			digitalWrite(pinCW_CCW1,   CW  );  // CW
			digitalWrite(pinCLK1,     LOW  );  // LOW 
			// digitalWrite(pinTEST11,     );  // we do not use this pin ( CMOS input on board )
		}
	}
	else
	{
	    boards=NOBRD;	
	}
}

//========================================================================
// Stepping motors MASTER API (boards==ONE)

#if defined TYPE1 
void // FULL_STEP, HALF_STEP_A, HALF_STEP_B, QUARTER_STEP
ROHM_Stepper::MODE  (int mode)
{
	switch (mode)
	{
		case FULL_STEP:
		digitalWrite(pinMODE00,0);
		digitalWrite(pinMODE10,0);
		break;
		
		case HALF_STEP_A:
		digitalWrite(pinMODE00,1);
		digitalWrite(pinMODE10,0);
		break;

		case HALF_STEP_B:
		digitalWrite(pinMODE00,0);
		digitalWrite(pinMODE10,1);
		break;		
		
		case QUARTER_STEP:
		digitalWrite(pinMODE00,1);
		digitalWrite(pinMODE10,1);
		break;

		default:
		break;		
	}
}
#endif

#if defined TYPE2 //
void // FULL_STEP, HALF_STEP, EIGHTH_STEP, SIXTEENTH_STEP
ROHM_Stepper::MODE  (int mode) 
{
	switch (mode)
	{
		case FULL_STEP:
		digitalWrite(pinMODE00,0);
		digitalWrite(pinMODE10,0);
		break;
		
		case HALF_STEP:
		digitalWrite(pinMODE00,1);
		digitalWrite(pinMODE10,0);	
		break;

		case EIGHTH_STEP:
		digitalWrite(pinMODE00,0);
		digitalWrite(pinMODE10,1);
		break;
		
		case SIXTEENTH_STEP:
		digitalWrite(pinMODE00,1);
		digitalWrite(pinMODE10,1);
		break;

		default:
		break;		
	}
}
#endif

#if defined TYPE3 //
void // FULL_STEP, HALF_STEP, QUARTER_STEP, SIXTEENTH_STEP
ROHM_Stepper::MODE  (int mode) 
{
	switch (mode)
	{
		case FULL_STEP:
		digitalWrite(pinMODE00,0);
		digitalWrite(pinMODE10,0);
		break;
		
		case HALF_STEP:
		digitalWrite(pinMODE00,1);
		digitalWrite(pinMODE10,0);	
		break;

		case QUARTER_STEP:
		digitalWrite(pinMODE00,0);
		digitalWrite(pinMODE10,1);
		break;
		
		case SIXTEENTH_STEP:
		digitalWrite(pinMODE00,1);
		digitalWrite(pinMODE10,1);
		break;

		default:
		break;		
	}
}
#endif

#if defined TYPE4 //
void // FULL_STEP, HALF_STEP, QUARTER_STEP, EIGHTH_STEP
ROHM_Stepper::MODE  (int mode) 
{
	switch (mode)
	{
		case FULL_STEP:
		digitalWrite(pinMODE00,0);
		digitalWrite(pinMODE10,0);
		break;
		
		case HALF_STEP:
		digitalWrite(pinMODE00,1);
		digitalWrite(pinMODE10,0);	
		break;

		case QUARTER_STEP:
		digitalWrite(pinMODE00,0);
		digitalWrite(pinMODE10,1);
		break;
		
		case EIGHTH_STEP:
		digitalWrite(pinMODE00,1);
		digitalWrite(pinMODE10,1);
		break;

		default:
		break;		
	}
}
#endif

void //------------------------------------------------------------------
ROHM_Stepper::ENABLE(int en)  // ACTIVE/OPEN
{
	digitalWrite(pinENABLE0,en);
}

void //------------------------------------------------------------------
ROHM_Stepper::PS    (int  ps) // ACTIVE/RESET
{
	digitalWrite(pinPS0,ps);
}

void //------------------------------------------------------------------
ROHM_Stepper::CW_CCW(int dir) // CW or CCW
{
	digitalWrite(pinCW_CCW0,dir);
}

void //------------------------------------------------------------------
ROHM_Stepper::CLK   (int clk) // zero, one or more clocks 
{
	if (clk)
	{
	  steps=clk;
	  while (steps)
	  {
		if(boards)digitalWrite(pinCLK0,HIGH);
		delayMicroseconds(tHP);
		if(boards)digitalWrite(pinCLK0,LOW);
		delayMicroseconds(tHP);		
		steps--;
	  }
	}
}

//========================================================================
// Stepping motors SLAVE API (boards==TWO)

#if defined TYPE1
void 
ROHM_Stepper::MODE_S  (int mode)
{
	switch (mode)
	{
		case FULL_STEP:
		digitalWrite(pinMODE01,0);
		digitalWrite(pinMODE11,0);
		break;
		
		case HALF_STEP_A:
		digitalWrite(pinMODE01,1);
		digitalWrite(pinMODE11,0);
		break;

		case HALF_STEP_B:
		digitalWrite(pinMODE01,0);
		digitalWrite(pinMODE11,1);
		break;		
		
		case QUARTER_STEP:
		digitalWrite(pinMODE01,1);
	    	digitalWrite(pinMODE11,1);
		break;

		default:
		break;		
	};
}
#endif

#if defined TYPE2
void // FULL_STEP, HALF_STEP, EIGHTH_STEP, SIXTEENTH_STEP
ROHM_Stepper::MODE_S  (int mode)
{
	switch (mode)
	{
		case FULL_STEP:
		digitalWrite(pinMODE01,0);
		digitalWrite(pinMODE11,0);
		break;
		
		case HALF_STEP:
		digitalWrite(pinMODE01,1);
		digitalWrite(pinMODE11,0);
		break;

		case EIGHTH_STEP:
		digitalWrite(pinMODE01,0);
		digitalWrite(pinMODE11,1);
		break;		
		
		case SIXTEENTH_STEP:
		digitalWrite(pinMODE01,1);
		digitalWrite(pinMODE11,1);
		break;

		default:
		break;		
	};
}
#endif

#if defined TYPE3
void // FULL_STEP, HALF_STEP, QUARTER_STEP, SIXTEENTH_STEP
ROHM_Stepper::MODE_S  (int mode)
{
	switch (mode)
	{
		case FULL_STEP:
		digitalWrite(pinMODE01,0);
		digitalWrite(pinMODE11,0);
		break;
		
		case HALF_STEP:
		digitalWrite(pinMODE01,1);
		digitalWrite(pinMODE11,0);
		break;

		case QUARTER_STEP:
		digitalWrite(pinMODE01,0);
		digitalWrite(pinMODE11,1);
		break;		
		
		case SIXTEENTH_STEP:
		digitalWrite(pinMODE01,1);
		digitalWrite(pinMODE11,1);
		break;

		default:
		break;		
	};
}
#endif

#if defined TYPE4
void // FULL_STEP, HALF_STEP, QUARTER_STEP, EIGHTH_STEP
ROHM_Stepper::MODE_S  (int mode)
{
	switch (mode)
	{
		case FULL_STEP:
		digitalWrite(pinMODE01,0);
		digitalWrite(pinMODE11,0);
		break;
		
		case HALF_STEP:
		digitalWrite(pinMODE01,1);
		digitalWrite(pinMODE11,0);
		break;

		case QUARTER_STEP:
		digitalWrite(pinMODE01,0);
		digitalWrite(pinMODE11,1);
		break;		
		
		case EIGHTH_STEP:
		digitalWrite(pinMODE01,1);
		digitalWrite(pinMODE11,1);
		break;

		default:
		break;		
	};
}
#endif

void //------------------------------------------------------------------
ROHM_Stepper::ENABLE_S(int en)  // ACTIVE/OPEN
{
	digitalWrite(pinENABLE1,en);
}

void //------------------------------------------------------------------
ROHM_Stepper::PS_S    (int  ps) // ACTIVE/RESET
{
	digitalWrite(pinPS1,ps);
}
void //------------------------------------------------------------------
ROHM_Stepper::CW_CCW_S(int dir) // CW or CCW
{
	digitalWrite(pinCW_CCW1,dir);
}

void //------------------------------------------------------------------
ROHM_Stepper::CLK_S   (int clk) // zero, one or more clocks 
{
	if (clk)
	{
		steps_S=clk;
		while (steps_S)
		{
			digitalWrite(pinCLK1,HIGH);
			delayMicroseconds(tHP);
			digitalWrite(pinCLK1,LOW);
			delayMicroseconds(tHP);		
			steps_S--;
		}
	}
}
//========================================================================
// CLK Emulation API

void //--------------------------------------------------------------------
ROHM_Stepper::setCLK_HP(unsigned int hp) // setting up new value for CLK half period
{
	if (hp){tHP=hp;}else{tHP=HPdef;}
}

unsigned int //------------------------------------------------------------
ROHM_Stepper::getCLK_HP() // return current value of CLK half period
{
	return tHP; 
}

void //--------------------------------------------------------------------
ROHM_Stepper::setCLK_Hz(unsigned int Hz) // setting up CLK frequency
{
    unsigned int hp = ((double)500000/((double)Hz)); // half period in microseconds calculating
	setCLK_HP(hp);
}
unsigned int //------------------------------------------------------------
ROHM_Stepper::getCLK_Hz() // return current value of CLK in Hertz
{
	return (unsigned int)(((double)500000/(double)tHP)); 
}
//========================================================================
// Common

int //-------------------------------------------------------------------------
ROHM_Stepper::version(void) // return version of the lib 
{
	if (boards){return 110;}else{ return 0;}
}

#endif // ROHM_Steppers_h
